<?php

namespace App\Http\Controllers;

use App\Models\Transaksi;
use App\Models\TransaksiItem;
use App\Models\PembelianDetail;
use App\Models\RekamMedik;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;

class TransaksiController extends Controller
{
    public function index()
    {
        $transaksis = Transaksi::with('items.obat', 'user')->latest()->get();

        $rekamMediks = RekamMedik::whereHas('resep')
            ->whereDoesntHave('transaksi')
            ->latest()
            ->get();

        return view('karyawan.transaksi.index', compact('transaksis', 'rekamMediks'));
    }

    public function create()
    {
        $obats = PembelianDetail::with('masterObat')
            ->whereHas('masterObat', fn($q) => $q->where('status', 'aktif'))
            ->get();

        return view('karyawan.transaksi.create', compact('obats'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'tanggal_transaksi' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.obat_id' => 'required|exists:pembelian_details,id',
            'items.*.jumlah' => 'required|integer|min:1',
        ]);

        DB::transaction(function () use ($request) {
            $transaksi = Transaksi::create([
                'user_id' => Auth::id(),
                'tanggal_transaksi' => $request->tanggal_transaksi,
                'total' => 0,
            ]);

            $total = 0;

            foreach ($request->items as $item) {
                $obat = PembelianDetail::findOrFail($item['obat_id']);
                $subtotal = $obat->harga_jual * $item['jumlah'];

                // Kurangi stok
                $obat->decrement('jumlah', $item['jumlah']);

                // Simpan item transaksi
                TransaksiItem::create([
                    'transaksi_id' => $transaksi->id,
                    'obat_id' => $obat->master_obat_id,
                    'pembelian_detail_id' => $obat->id,
                    'jumlah' => $item['jumlah'],
                    'harga_satuan' => $obat->harga_jual,
                    'subtotal' => $subtotal,
                    'nama_obat' => $obat->masterObat->nama,
                    'satuan' => $obat->satuan,
                ]);

                $total += $subtotal;
            }

            $transaksi->update(['total' => $total]);
        });

        return redirect()->route('karyawan.transaksi.index')
            ->with('success', 'Transaksi berhasil disimpan.');
    }

    public function listRekamMedik()
    {
        $rekamMediks = RekamMedik::with('dokter', 'resep')
            ->whereHas('resep')
            ->whereDoesntHave('transaksi')
            ->latest()
            ->get();

        return view('karyawan.transaksi.rekam_medik', compact('rekamMediks'));
    }

    public function createFromRekamMedik($rekamMedikId)
    {
        $rekamMedik = RekamMedik::with('resep.obats')->findOrFail($rekamMedikId);

        return view('karyawan.transaksi.rekam_medik_show', compact('rekamMedik'));
    }

    public function storeFromRekamMedik(Request $request, $rekamMedikId)
    {
        $rekamMedik = RekamMedik::with('resep.obats')->findOrFail($rekamMedikId);

        DB::transaction(function () use ($rekamMedik) {
            $transaksi = Transaksi::create([
                'user_id' => Auth::id(),
                'rekam_medik_id' => $rekamMedik->id,
                'tanggal_transaksi' => now(),
                'total' => 0,
            ]);

            $total = 0;

            foreach ($rekamMedik->resep->obats as $obat) {
                // cari stok di pembelian_detail sesuai master_obat_id
                $detail = PembelianDetail::where('master_obat_id', $obat->id)
                    ->where('jumlah', '>', 0)
                    ->firstOrFail();

                $jumlah = $obat->pivot->jumlah; // dari resep
                $subtotal = $detail->harga_jual * $jumlah;

                // kurangi stok
                $detail->decrement('jumlah', $jumlah);

                // simpan item transaksi
                TransaksiItem::create([
                    'transaksi_id' => $transaksi->id,
                    'obat_id' => $detail->master_obat_id,
                    'pembelian_detail_id' => $detail->id,
                    'jumlah' => $jumlah,
                    'harga_satuan' => $detail->harga_jual,
                    'subtotal' => $subtotal,
                    'nama_obat' => $detail->masterObat->nama,
                    'satuan' => $detail->satuan,
                ]);

                $total += $subtotal;
            }

            $transaksi->update(['total' => $total]);
        });

        return redirect()->route('karyawan.transaksi.index')
            ->with('success', 'Transaksi dari rekam medik berhasil ditebus.');
    }

    public function show($id)
    {
        $transaksi = Transaksi::with('items.obat', 'user')->findOrFail($id);
        return view('karyawan.transaksi.show', compact('transaksi'));
    }

    public function riwayatRekamMedik()
    {
        $transaksis = Transaksi::with('items.obat', 'user', 'rekamMedik')
            ->whereNotNull('rekam_medik_id')
            ->latest()
            ->get();

        return view('karyawan.transaksi.riwayat_rekam_medik', compact('transaksis'));
    }

    public function printRekamMedik($id)
    {
        $transaksi = Transaksi::with(
            'items.obat',
            'user',
            'rekamMedik.resep.dokter',
            'rekamMedik.resep.resepObats.obat'
        )->findOrFail($id);

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView(
            'karyawan.transaksi.pdf_rekam_medik',
            compact('transaksi')
        )->setPaper('a4', 'portrait');

        return $pdf->download('rekam-medik-'.$transaksi->id.'.pdf');
    }

}
