<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\MasterObat;
use App\Models\Resep;
use App\Models\Obat;
use App\Models\ResepObat;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;

class ResepController extends Controller
{
    public function index(Request $request)
    {
        $query = Resep::where('dokter_id', Auth::id())
            ->with('resepObats.obat');

        if ($request->has('q') && $request->q != '') {
            $query->where('kode_resep', 'like', '%' . $request->q . '%');
        }

        if ($request->filled('obat')) {
            $query->whereHas('resepObats.obat', function ($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->obat . '%');
            });
        }

        $reseps = $query->latest()->paginate(10);

        return view('dokter.reseps.index', compact('reseps'));
    }

    public function create()
    {
        $obats = \App\Models\MasterObat::where('status', 'aktif')->get(); // tampilkan 10 per halaman
        return view('dokter.reseps.create', compact('obats'));
    }

   public function store(Request $request)
    {
        $request->validate([
            'kode_resep' => 'required|array',
            'kode_resep.*' => 'required|string|distinct',
            'obat_id' => 'required|array',
            'dosis' => 'required|array',
            'aturan_pakai' => 'required|array',
        ]);

        DB::beginTransaction();

        try {
            foreach ($request->kode_resep as $index => $kode) {
                // Simpan resep
                $resep = Resep::create([
                    'dokter_id' => Auth::id(),
                    'kode_resep' => $kode,
                ]);

                // Ambil obat-obat untuk resep ini berdasarkan index
                $obatIds = $request->obat_id[$index] ?? [];
                $dosisList = $request->dosis[$index] ?? [];
                $aturanList = $request->aturan_pakai[$index] ?? [];

                foreach ($obatIds as $i => $obatId) {
                    if (!$obatId) continue;

                    ResepObat::create([
                        'resep_id' => $resep->id,
                        'obat_id' => $obatId,
                        'dosis' => $dosisList[$i] ?? '',
                        'aturan_pakai' => $aturanList[$i] ?? '',
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('dokter.reseps.index')->with('success', 'Semua resep berhasil disimpan!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'Terjadi kesalahan saat menyimpan: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        $resep = Resep::with('resepObats.obat')
            ->where('dokter_id', Auth::id())
            ->findOrFail($id);

        $obats = Obat::all();
        return view('dokter.reseps.show', compact('resep', 'obats'));
    }


   public function update(Request $request, $id)
    {
        $request->validate([
            'kode_resep' => 'required|string|max:255',
        ]);

        $resep = Resep::where('dokter_id', Auth::id())->findOrFail($id);
        $resep->update([
            'kode_resep' => $request->kode_resep,
        ]);

        return redirect()->route('dokter.reseps.index')->with('success', 'Resep berhasil diperbarui.');
    }


    public function edit($id)
    {
        $resep = Resep::with('resepObats.obat')->findOrFail($id);
        $obats = Obat::all();
        return view('dokter.reseps.edit', compact('resep', 'obats'));
    }

    public function destroy($id)
    {
        $resep = Resep::findOrFail($id);
        $resep->resepObats()->delete(); // hapus relasi dulu
        $resep->delete();

        return redirect()->route('dokter.reseps.index')->with('success', 'Resep berhasil dihapus.');
    }

}
