<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Obat;
use App\Models\Kategori;


class ObatController extends Controller
{
    public function index(Request $request)
    {
        $query = Obat::with('kategori')
            ->where('status', 'aktif')
            ->where(function ($q) {
            $q->whereNull('expired_at')
            ->orWhereDate('expired_at', '>=', now());
        })->latest();

        if ($request->has('cari')) {
            $query->where('nama', 'like', '%' . $request->cari . '%');
        }

        if ($request->filled('obat_id')) {
            $query->where('id', $request->obat_id); // cari berdasarkan ID dari select2
        }

        $obats = \App\Models\Obat::with(['kategori', 'pembelians.supplier'])
        ->when($request->obat_id, function ($query) use ($request) {
            $query->where('id', $request->obat_id);
        })
        ->paginate(10); 

        //$obats = $query->paginate(10);

        //if (Auth::user()->role === 'karyawan') {
            //if ($request->ajax()) {
                //return view('karyawan.obat.data', compact('obats'))->render();
            //}

            //return view('karyawan.obat.index', compact('obats'));
        //}

        return view('admin.obat.index', compact('obats'));
    }

    public function create()
    {
        $kategoris = Kategori::all();
        return view('admin.obat.create', compact('kategoris'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            'kategori_id' => 'required',
            'stok' => 'required|numeric',
            'harga' => 'required|numeric',
        ]);

        $data = $request->all();
        $data['status'] = 'aktif'; // default aktif
        Obat::create($data);

        return redirect()->route('admin.obats.index')->with('success', 'Obat berhasil ditambahkan.');
    }

    public function edit($id)
    {
        $obat = Obat::findOrFail($id);
        $kategoris = Kategori::all();

        return view('admin.obat.edit', compact('obat', 'kategoris'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'nama' => 'required',
            'kategori_id' => 'required',
            'stok' => 'required|numeric',
            'harga' => 'required|numeric',
        ]);

        $obat = Obat::findOrFail($id);
        $obat->update($request->all());

        return redirect()->route('admin.obats.index')->with('success', 'Obat berhasil diperbarui.');
    }

    public function show($id)
    {
        $obat = Obat::findOrFail($id);
        return view('admin.obat.show', compact('obat'));
    }

    public function destroy($id)
    {
        $obat = Obat::findOrFail($id);
        $obat->delete();

        return redirect()->route('admin.obats.index')->with('success', 'Obat berhasil dihapus.');
    }

    public function kadaluarsa()
    {
        $obats = Obat::whereDate('expired_at', '<', now())
                ->orWhere('status', 'tidak aktif')
                ->get();

        return view('admin.obat.kadaluarsa', compact('obats'));
    }

    public function stokMenipis()
    {
        $obats = Obat::where('stok', '<=', 10)
            ->where('status', 'aktif')
            ->get();

        return view('admin.obat.stok_menipis', compact('obats'));
    }

    public function ubahStatus($id)
    {
        $obat = Obat::findOrFail($id);
        $obat->update(['status' => 'aktif']);

        return redirect()->route('admin.obat.kadaluarsa')->with('success', 'Status obat berhasil diubah menjadi aktif.');
    }

    public function aktifkan($id)
    {
        $obat = Obat::findOrFail($id);

        $obat->update([
            'status' => 'aktif',
            'expired_at' => now()->addYear(), // default 1 tahun ke depan
        ]);

        return redirect()->route('admin.obats.kadaluarsa')->with('success', 'Obat berhasil diaktifkan kembali.');
    }

    public function nonAktifkan($id)
    {
        $obat = Obat::findOrFail($id);
        $obat->update(['status' => 'tidak aktif']);

        return redirect()->route('admin.obats.kadaluarsa')->with('success', 'Obat berhasil dinonaktifkan.');
    }

   public function search(Request $request)
    {
        $search = $request->get('q');
        $obats = Obat::where('nama', 'like', "%{$search}%")
                    ->select('id', 'nama', 'harga', 'stok')
                    ->limit(10)
                    ->get();

        return response()->json($obats);
    }
}
