<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MasterObat;
use App\Models\Kategori;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MasterObatController extends Controller
{
    public function index()
    {
        $obats = MasterObat::with('kategori')->latest()->paginate(10);
        return view('admin.master_obat.index', compact('obats'));
    }

    public function create()
    {
        $kategoris = Kategori::all();
        return view('admin.master_obat.create', compact('kategoris'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'kategori_id' => 'required|exists:kategoris,id',
        ]);

        MasterObat::create([
            'nama' => $request->nama,
            'kategori_id' => $request->kategori_id
        ]);

        return redirect()->route('admin.master_obat.index')
            ->with('success', 'Master Obat berhasil ditambahkan.');
    }

    public function edit($id)
    {
        $obat = MasterObat::findOrFail($id);
        $kategoris = Kategori::all();
        return view('admin.master_obat.edit', compact('obat', 'kategoris'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'kategori_id' => 'required|exists:kategoris,id',
        ]);

        $obat = MasterObat::findOrFail($id);
        $obat->update([
            'nama' => $request->nama,
            'kategori_id' => $request->kategori_id
        ]);

        return redirect()->route('admin.master_obat.index')
            ->with('success', 'Master Obat berhasil diperbarui.');
    }

    public function destroy($id)
    {
        MasterObat::findOrFail($id)->delete();
        return redirect()->route('admin.master_obat.index')
            ->with('success', 'Master Obat berhasil dihapus.');
    }

    public function show($id)
    {
        $obat = MasterObat::with(['kategori', 'pembelianDetails.pembelian.supplier'])
            ->findOrFail($id);
        return view('admin.master_obat.show', compact('obat'));
    }

    public function kadaluarsa()
    {
        $today = now()->toDateString();

        $masterObats = DB::table('pembelian_details')
            ->join('master_obats', 'pembelian_details.master_obat_id', '=', 'master_obats.id')
            ->leftJoin('kategoris', 'master_obats.kategori_id', '=', 'kategoris.id')
            ->select(
                'master_obats.id',
                'master_obats.nama',
                'master_obats.status',
                'kategoris.nama as kategori_nama',
                DB::raw('MIN(pembelian_details.expired_at) as expired_at'),
                DB::raw('SUM(pembelian_details.jumlah) as total_stok')
            )
            // Hanya tampilkan jika TIDAK ADA batch yang masih berlaku
            ->whereNotExists(function($query) use ($today) {
                $query->select(DB::raw(1))
                    ->from('pembelian_details as pd')
                    ->whereRaw('pd.master_obat_id = master_obats.id')
                    ->whereDate('pd.expired_at', '>=', $today);
            })
            ->groupBy('master_obats.id', 'master_obats.nama', 'master_obats.status', 'kategoris.nama')
            ->orderBy('expired_at', 'asc')
            ->get()
            ->map(function($item) {
                $item->kategori = (object)['nama' => $item->kategori_nama];
                return $item;
            });

        return view('admin.master_obat.kadaluarsa', compact('masterObats'));
    }

    public function aktifkan($id)
    {
        $obat = MasterObat::findOrFail($id);
        $obat->update(['status' => 'aktif']);
        return redirect()->route('admin.master_obat.kadaluarsa')
                        ->with('success', 'Obat berhasil diaktifkan kembali.');
    }

    public function nonaktifkan($id)
    {
        $obat = MasterObat::findOrFail($id);
        $obat->update(['status' => 'tidak aktif']);
        return redirect()->route('admin.master_obat.kadaluarsa')
            ->with('success', 'Obat berhasil dinonaktifkan.');
    }

    public function stokMenipis()
    {
        // Tampilkan obat dengan status aktif atau tidak aktif, tapi stok_tersedia < 10
        $today = now()->toDateString();

        $obats = MasterObat::with(['kategori'])
            ->whereIn('status', ['aktif', 'tidak aktif'])
            ->withSum(['pembelianDetails as stok_tersedia' => function ($query) use ($today) {
                $query->where(function ($q) use ($today) {
                    $q->whereNull('expired_at')
                    ->orWhere('expired_at', '>=', $today);
                });
            }], 'jumlah')
            ->havingRaw('COALESCE(stok_tersedia, 0) < 10')
            ->get();

        // Perbaiki stok_tersedia untuk kasus stok 0 tapi masih ada pembelian valid
        foreach ($obats as $obat) {
            if ($obat->stok_tersedia === 0) {
                // Hitung stok total tanpa filter expired_at
                $totalStok = $obat->pembelianDetails()->sum('jumlah');
                if ($totalStok > 0) {
                    // Override stok_tersedia dengan totalStok
                    $obat->setAttribute('stok_tersedia', $totalStok);
                }
            }
        }

        return view('admin.master_obat.stok_menipis', compact('obats'));
    }    
}
