<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MasterObat;
use App\Models\Transaksi;
use App\Models\Supplier;
use App\Models\TransaksiItem;
use App\Models\PembelianDetail;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    public function dashboard()
{
    $totalObat = MasterObat::count();
    $today = now()->toDateString();

    // Hitung stok menipis: MasterObat dengan stok tersedia <= 10
    $stokMenipis = MasterObat::whereIn('status', ['aktif', 'tidak aktif'])
        ->withSum(['pembelianDetails as stok_tersedia' => function ($query) use ($today) {
            $query->where(function ($q) use ($today) {
                $q->whereNull('expired_at')
                  ->orWhere('expired_at', '>=', $today);
            });
        }], 'jumlah')
        ->havingRaw('COALESCE(stok_tersedia, 0) <= 10')
        ->count();

    // Hitung kadaluarsa: MasterObat yang tidak memiliki batch aktif
    $kadaluarsaCount = DB::table('master_obats')
        ->whereNotExists(function($query) use ($today) {
            $query->select(DB::raw(1))
                ->from('pembelian_details as pd')
                ->whereRaw('pd.master_obat_id = master_obats.id')
                ->whereDate('pd.expired_at', '>=', $today);
        })
        ->count();

    $totalTransaksi = Transaksi::count();
    $totalSupplier = Supplier::count();

    // Obat kadaluarsa: MasterObat tanpa batch aktif
    $obatKadaluarsa = DB::table('pembelian_details')
        ->join('master_obats', 'pembelian_details.master_obat_id', '=', 'master_obats.id')
        ->leftJoin('kategoris', 'master_obats.kategori_id', '=', 'kategoris.id')
        ->select(
            'master_obats.id',
            'master_obats.nama',
            'master_obats.status',
            'kategoris.nama as kategori_nama',
            DB::raw('MIN(pembelian_details.expired_at) as expired_at'),
            DB::raw('SUM(pembelian_details.jumlah) as stok')
        )
        ->whereNotExists(function($query) use ($today) {
            $query->select(DB::raw(1))
                ->from('pembelian_details as pd')
                ->whereRaw('pd.master_obat_id = master_obats.id')
                ->whereDate('pd.expired_at', '>=', $today);
        })
        ->groupBy('master_obats.id', 'master_obats.nama', 'master_obats.status', 'kategoris.nama')
        ->orderBy('expired_at', 'asc')
        ->get()
        ->map(function($item) {
            $item->kategori = (object)['nama' => $item->kategori_nama];
            return $item;
        });

    // Obat stok menipis
    $obatStokMenipis = DB::table('master_obats')
        ->leftJoin('kategoris', 'master_obats.kategori_id', '=', 'kategoris.id')
        ->leftJoin('pembelian_details', 'master_obats.id', '=', 'pembelian_details.master_obat_id')
        ->select(
            'master_obats.id',
            'master_obats.nama',
            'master_obats.status',
            'kategoris.nama as kategori_nama',
            DB::raw('COALESCE(SUM(CASE WHEN pembelian_details.expired_at IS NULL OR pembelian_details.expired_at >= "' . $today . '" THEN pembelian_details.jumlah ELSE 0 END), 0) as stok'),
            DB::raw('MIN(pembelian_details.expired_at) as expired_at')
        )
        ->whereIn('master_obats.status', ['aktif', 'tidak aktif'])
        ->groupBy('master_obats.id', 'master_obats.nama', 'master_obats.status', 'kategoris.nama')
        ->havingRaw('stok <= 10')
        ->get()
        ->map(function($item) {
            $item->kategori = (object)['nama' => $item->kategori_nama];
            return $item;
        });

    // Top obat berdasarkan stok tersedia
    $topObats = MasterObat::with(['kategori'])
        ->withSum(['pembelianDetails as total_stok' => function ($query) use ($today) {
            $query->where(function ($q) use ($today) {
                $q->whereNull('expired_at')
                  ->orWhere('expired_at', '>=', $today);
            });
        }], 'jumlah')
        ->orderByDesc('total_stok')
        ->limit(5)
        ->get()
        ->map(function($obat) {
            return (object)[
                'nama' => $obat->nama,
                'total' => $obat->total_stok ?? 0
            ];
        });

    $today = Carbon::today();

    $omzetHariIni = Transaksi::whereDate('tanggal_transaksi', $today)->sum('total');

    $totalObatTerjual = TransaksiItem::whereHas('transaksi', fn($q) =>
        $q->whereDate('tanggal_transaksi', $today)
    )->sum('jumlah');

    $jenisObatTerjual = TransaksiItem::whereHas('transaksi', fn($q) =>
        $q->whereDate('tanggal_transaksi', $today)
    )->distinct('obat_id')->count('obat_id');

    $detailObat = TransaksiItem::with('obat')
        ->whereHas('transaksi', fn($q) =>
            $q->whereDate('tanggal_transaksi', $today)
        )
        ->select('obat_id', DB::raw('SUM(jumlah) as total'))
        ->groupBy('obat_id')
        ->get()
        ->filter(fn($item) => $item->obat !== null);

    $penjualanBulanan = Transaksi::select(
        DB::raw('DATE(tanggal_transaksi) as tanggal'),
        DB::raw('SUM(total) as total')
    )
    ->whereMonth('tanggal_transaksi', now()->month)
    ->groupBy(DB::raw('DATE(tanggal_transaksi)'))
    ->orderBy('tanggal')
    ->get();

    return view('admin.dashboard', [
        'totalObat' => $totalObat,
        'stokMenipis' => $stokMenipis,
        'kadaluarsa' => $kadaluarsaCount, // hanya jumlah
        'topObats' => $topObats,
        'omzetHariIni' => $omzetHariIni,
        'totalObatTerjual' => $totalObatTerjual,
        'jenisObatTerjual' => $jenisObatTerjual,
        'detailObat' => $detailObat,
        'penjualanBulanan' => $penjualanBulanan,
        'obatKadaluarsa' => $obatKadaluarsa, // untuk ditampilkan di tabel
        'obatStokMenipis' => $obatStokMenipis,
        'totalTransaksi' => $totalTransaksi,
        'totalSupplier' => $totalSupplier,
        'today' => $today,
    ]);
}
}
