<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MasterObat;
use App\Models\PembelianDetail;
use App\Models\Transaksi;
use App\Models\TransaksiItem;
use App\Models\RekamMedik; // relasi dari dokter
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TransaksiKaryawanController extends Controller
{
    /**
     * List transaksi karyawan (riwayat)
     */
    public function index(Request $request)
    {
        $query = Transaksi::with('items.masterObat')
            ->where('user_id', Auth::id())
            ->orderBy('tanggal_transaksi', 'desc');

        if ($request->filled('tanggal')) {
            $query->whereDate('tanggal_transaksi', $request->tanggal);
        }

        $transaksis = $query->paginate(10);

        return view('karyawan.transaksi.index', compact('transaksis'));
    }

    /**
     * Halaman buat transaksi baru
     */
    public function create()
    {
        return view('karyawan.transaksi.create');
    }

    /**
     * Cari obat untuk transaksi langsung (AJAX)
     */
    public function searchObat(Request $request)
    {
        $search = $request->query('q');

        $obats = MasterObat::with(['pembelianDetails' => function ($q) {
            $q->latest();
        }])
        ->where('nama', 'like', "%{$search}%")
        ->limit(10)
        ->get()
        ->map(function ($obat) {
            $harga = $obat->pembelianDetails->first()->harga_jual ?? 0;
            $stok = $obat->pembelianDetails->sum('jumlah');
            return [
                'id' => $obat->id,
                'nama' => $obat->nama,
                'harga' => $harga,
                'stok' => $stok,
            ];
        });

        return response()->json($obats);
    }

    /**
     * Daftar rekam medik dari dokter yang belum ditebus
     */
    public function rekamMedikList()
    {
        $rekamMediks = RekamMedik::with('resep.items.obat')
            ->where('status_tebus', 'belum')
            ->latest()
            ->paginate(10);

        return view('karyawan.transaksi.rekam-medik', compact('rekamMediks'));
    }

    /**
     * Tebus obat dari rekam medik
     */
    public function tebusObat($rekamMedikId)
    {
        $rekamMedik = RekamMedik::with('resep.items.obat')->findOrFail($rekamMedikId);

        DB::beginTransaction();
        try {
            $total = 0;
            $items = [];

            foreach ($rekamMedik->resep->items as $item) {
                $pembelian = PembelianDetail::where('master_obat_id', $item->obat->id)
                    ->orderBy('created_at', 'desc')
                    ->first();

                if (!$pembelian) {
                    DB::rollBack();
                    return back()->with('error', "Data pembelian untuk {$item->obat->nama} tidak ditemukan.");
                }

                $stok = PembelianDetail::where('master_obat_id', $item->obat->id)->sum('jumlah');

                if ($stok < $item->jumlah) {
                    DB::rollBack();
                    return back()->with('error', "Stok obat {$item->obat->nama} tidak mencukupi.");
                }

                $subtotal = $pembelian->harga_jual * $item->jumlah;
                $total += $subtotal;

                $items[] = [
                    'master_obat' => $item->obat,
                    'jumlah' => $item->jumlah,
                    'harga_satuan' => $pembelian->harga_jual,
                    'subtotal' => $subtotal,
                ];
            }

            $transaksi = Transaksi::create([
                'user_id' => Auth::id(),
                'tipe' => 'resep',
                'rekam_medik_id' => $rekamMedik->id,
                'tanggal_transaksi' => now(),
                'total' => $total,
            ]);

            foreach ($items as $item) {
                TransaksiItem::create([
                    'transaksi_id' => $transaksi->id,
                    'master_obat_id' => $item['master_obat']->id,
                    'jumlah' => $item['jumlah'],
                    'harga_satuan' => $item['harga_satuan'],
                    'subtotal' => $item['subtotal'],
                ]);

                $this->kurangiStok($item['master_obat']->id, $item['jumlah']);
            }

            $rekamMedik->update(['status_tebus' => 'sudah']);

            DB::commit();
            return redirect()->route('karyawan.transaksi.show', $transaksi->id)
                ->with('success', 'Resep berhasil ditebus!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Simpan transaksi langsung
     */
    public function store(Request $request)
    {
        $request->validate([
            'obat_id' => 'required|array',
            'obat_id.*' => 'required|exists:master_obats,id',
            'jumlah' => 'required|array',
            'jumlah.*' => 'required|integer|min:1',
        ]);

        DB::beginTransaction();
        try {
            $total = 0;
            $items = [];

            foreach ($request->obat_id as $i => $obatId) {
                $masterObat = MasterObat::findOrFail($obatId);
                $pembelian = PembelianDetail::where('master_obat_id', $masterObat->id)
                    ->orderBy('created_at', 'desc')
                    ->first();

                if (!$pembelian) {
                    DB::rollBack();
                    return back()->with('error', "Data pembelian untuk {$masterObat->nama} tidak ditemukan.");
                }

                $jumlah = $request->jumlah[$i];
                $stok = PembelianDetail::where('master_obat_id', $masterObat->id)->sum('jumlah');

                if ($stok < $jumlah) {
                    DB::rollBack();
                    return back()->with('error', "Stok obat {$masterObat->nama} tidak mencukupi.");
                }

                $subtotal = $pembelian->harga_jual * $jumlah;
                $total += $subtotal;

                $items[] = [
                    'master_obat' => $masterObat,
                    'jumlah' => $jumlah,
                    'harga_satuan' => $pembelian->harga_jual,
                    'subtotal' => $subtotal,
                ];
            }

            $transaksi = Transaksi::create([
                'user_id' => Auth::id(),
                'tipe' => 'langsung',
                'tanggal_transaksi' => now(),
                'total' => $total,
            ]);

            foreach ($items as $item) {
                TransaksiItem::create([
                    'transaksi_id' => $transaksi->id,
                    'master_obat_id' => $item['master_obat']->id,
                    'jumlah' => $item['jumlah'],
                    'harga_satuan' => $item['harga_satuan'],
                    'subtotal' => $item['subtotal'],
                ]);

                $this->kurangiStok($item['master_obat']->id, $item['jumlah']);
            }

            DB::commit();
            return redirect()->route('karyawan.transaksi.show', $transaksi->id)
                ->with('success', 'Transaksi berhasil!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Kurangi stok dengan metode FIFO
     */
    protected function kurangiStok($masterObatId, $jumlahDikurangi)
    {
        $details = PembelianDetail::where('master_obat_id', $masterObatId)
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($details as $detail) {
            if ($jumlahDikurangi <= 0) break;

            if ($detail->jumlah >= $jumlahDikurangi) {
                $detail->jumlah -= $jumlahDikurangi;
                $detail->save();
                $jumlahDikurangi = 0;
            } else {
                $jumlahDikurangi -= $detail->jumlah;
                $detail->jumlah = 0;
                $detail->save();
            }
        }
    }

    /**
     * Detail transaksi
     */
    public function show($id)
    {
        $transaksi = Transaksi::with('items.masterObat')->findOrFail($id);
        return view('karyawan.transaksi.show', compact('transaksi'));
    }
}
