<?php

namespace App\Http\Controllers;

use App\Models\Pembelian;
use App\Models\PembelianDetail;
use App\Models\Supplier;
use App\Models\MasterObat;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;


class PembelianController extends Controller
{
    public function index(Request $request)
    {
        $query = Pembelian::with(['supplier', 'details.masterObat']);

        $selectedObat = $request->get('master_obat_id');

        // Pencarian berdasarkan nama obat
        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->whereHas('details.masterObat', function ($q) use ($search) {
                $q->where('nama', 'like', "%{$search}%");
            });
        }

        $pembelians = $query->latest()->paginate(10);

        return view('admin.pembelian.index', compact('pembelians', 'selectedObat'));
    }
    
    public function create()
    {
        $suppliers = Supplier::all();
        $masterObats = MasterObat::all();
        return view('admin.pembelian.create', compact('suppliers', 'masterObats'));
    }

    public function store(Request $request)
    {
        // Validasi input
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'tanggal_pembelian' => 'required|date',
            'obat' => 'required|array|min:1',
        ]);

        // Validasi untuk setiap item obat
        foreach ($request->obat as $index => $item) {
            $request->validate([
                "obat.{$index}.master_obat_id" => 'required|exists:master_obats,id',
                "obat.{$index}.kode_barcode" => 'nullable|string|max:100',
                "obat.{$index}.jumlah" => 'required|integer|min:1',
                "obat.{$index}.satuan" => 'required|string',
                "obat.{$index}.harga_beli" => 'required|numeric|min:0',
                "obat.{$index}.harga_jual" => 'required|numeric|min:0',
                "obat.{$index}.expired_at" => 'nullable|date',
            ]);
        }

        DB::transaction(function () use ($request, $validated) {
            // Hitung total pembayaran
            $totalPembayaran = 0;
            foreach ($request->obat as $item) {
                $totalPembayaran += $item['jumlah'] * $item['harga_beli'];
            }

            // Simpan pembelian utama
            $pembelian = Pembelian::create([
                'supplier_id' => $validated['supplier_id'],
                'tanggal_pembelian' => $validated['tanggal_pembelian'],
                'total_pembayaran' => $totalPembayaran,
            ]);

            // Simpan detail pembelian
            foreach ($request->obat as $item) {
                PembelianDetail::create([
                    'pembelian_id' => $pembelian->id,
                    'master_obat_id' => $item['master_obat_id'],
                    'kode_barcode' => $item['kode_barcode'] ?? null,
                    'jumlah' => $item['jumlah'],
                    'satuan' => $item['satuan'],
                    'harga_beli' => $item['harga_beli'],
                    'harga_jual' => $item['harga_jual'],
                    'expired_at' => $item['expired_at'] ?? null,
                    'subtotal' => $item['jumlah'] * $item['harga_beli'],
                ]);
            }
        });

        return redirect()->route('admin.pembelian.index')->with('success', 'Data pembelian berhasil disimpan.');
    }

    public function show($id)
    {
        $pembelian = Pembelian::with(['supplier', 'details.masterObat'])->findOrFail($id);
        return view('admin.pembelian.show', compact('pembelian'));
    }

    public function destroy($id)
    {
        $pembelian = Pembelian::findOrFail($id);
        $pembelian->delete();

        return back()->with('success', 'Data pembelian berhasil dihapus');
    }

    public function cetak($id)
    {
        $pembelian = Pembelian::with(['supplier', 'details.masterObat'])->findOrFail($id);

        $pdf = PDF::loadView('admin.pembelian.cetak', compact('pembelian'))
                ->setPaper('a4', 'portrait');

        return $pdf->download('pembelian_'.$pembelian->id.'.pdf');
    }

    
}
